<?php
// public/settings.php

require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/Database.php';

requireLogin();

$userId = $_SESSION['user_id'];
$db = Database::getInstance();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRFToken($_POST['csrf_token'] ?? '');
    
    // Prepare settings array from POST data
    $settings = [
        'enabled' => isset($_POST['enabled']) ? 1 : 0,
        'show_notifications' => isset($_POST['show_notifications']) ? 1 : 0,
        'send_emails' => isset($_POST['send_emails']) ? 1 : 0,
        'notification_cooldown' => intval($_POST['notification_cooldown'] ?? 30),
        'rsi_check_interval' => intval($_POST['rsi_check_interval'] ?? 70),
        'order_book_interval' => intval($_POST['order_book_interval'] ?? 300),
        'liquidation_interval' => intval($_POST['liquidation_interval'] ?? 120),
        'rsi7_15m_enabled' => isset($_POST['rsi7_15m_enabled']) ? 1 : 0,
        'rsi7_1d_enabled' => isset($_POST['rsi7_1d_enabled']) ? 1 : 0,
        'rsi_threshold' => intval($_POST['rsi_threshold'] ?? 90),
        'price_alerts_enabled' => isset($_POST['price_alerts_enabled']) ? 1 : 0,
        'price_break_threshold' => floatval($_POST['price_break_threshold'] ?? 0.4),
        'price_increase_threshold' => intval($_POST['price_increase_threshold'] ?? 14),
        'support_resistance_alerts_enabled' => isset($_POST['support_resistance_alerts_enabled']) ? 1 : 0,
        'support_resistance_threshold' => floatval($_POST['support_resistance_threshold'] ?? 0.4),
        'order_book_alerts_enabled' => isset($_POST['order_book_alerts_enabled']) ? 1 : 0,
        'order_book_depth' => intval($_POST['order_book_depth'] ?? 50),
        'order_book_bearish_threshold' => intval($_POST['order_book_bearish_threshold'] ?? -15),
        'order_book_bullish_threshold' => intval($_POST['order_book_bullish_threshold'] ?? 15),
        'min_volume' => intval($_POST['min_volume'] ?? 1000000),
        'min_order_book_volume' => intval($_POST['min_order_book_volume'] ?? 10000000),
        'liquidation_alert_threshold' => intval($_POST['liquidation_alert_threshold'] ?? 1000000),
        'multi_condition_alerts_enabled' => isset($_POST['multi_condition_alerts_enabled']) ? 1 : 0,
        'volume_spike_threshold' => floatval($_POST['volume_spike_threshold'] ?? 5),
        'liquidation_cluster_threshold' => intval($_POST['liquidation_cluster_threshold'] ?? 3),
        'funding_rate_alert_threshold' => floatval($_POST['funding_rate_alert_threshold'] ?? -0.01),
        'oi_change_threshold' => intval($_POST['oi_change_threshold'] ?? 20),
        'whale_order_threshold' => intval($_POST['whale_order_threshold'] ?? 1000000),
        'breakout_volume_multiplier' => floatval($_POST['breakout_volume_multiplier'] ?? 2),
        'second_rejection_alerts_enabled' => isset($_POST['second_rejection_alerts_enabled']) ? 1 : 0,
        'resistance_rejection_threshold' => floatval($_POST['resistance_rejection_threshold'] ?? 0.5),
        'two_percent_drop_alerts_enabled' => isset($_POST['two_percent_drop_alerts_enabled']) ? 1 : 0,
        'coin_tracking_enabled' => isset($_POST['coin_tracking_enabled']) ? 1 : 0,
        'coin_tracking_interval' => intval($_POST['coin_tracking_interval'] ?? 60),
        'pullback_threshold' => floatval($_POST['pullback_threshold'] ?? 2),
        'reversal_threshold' => floatval($_POST['reversal_threshold'] ?? 3),
        'tracked_volume_spike_multiplier' => floatval($_POST['tracked_volume_spike_multiplier'] ?? 3),
        'resistance_support_levels' => intval($_POST['resistance_support_levels'] ?? 3),
        'track_coin_history_days' => intval($_POST['track_coin_history_days'] ?? 7),
        'directional_volume_spike_enabled' => isset($_POST['directional_volume_spike_enabled']) ? 1 : 0,
        'buy_volume_spike_threshold' => floatval($_POST['buy_volume_spike_threshold'] ?? 3),
        'sell_volume_spike_threshold' => floatval($_POST['sell_volume_spike_threshold'] ?? 3),
        'volume_spike_candle_count' => intval($_POST['volume_spike_candle_count'] ?? 3),
        'volume_spike_min_price_move' => floatval($_POST['volume_spike_min_price_move'] ?? 1.5),
        'accumulation_distribution_enabled' => isset($_POST['accumulation_distribution_enabled']) ? 1 : 0,
        'pattern_recognition_enabled' => isset($_POST['pattern_recognition_enabled']) ? 1 : 0,
        'pattern_check_interval' => intval($_POST['pattern_check_interval'] ?? 300),
        'min_pattern_confidence' => intval($_POST['min_pattern_confidence'] ?? 60),
        'pattern_alert_cooldown' => intval($_POST['pattern_alert_cooldown'] ?? 3600),
        'enable_chart_patterns' => isset($_POST['enable_chart_patterns']) ? 1 : 0,
        'enable_candlestick_patterns' => isset($_POST['enable_candlestick_patterns']) ? 1 : 0,
        'enable_divergence_patterns' => isset($_POST['enable_divergence_patterns']) ? 1 : 0,
        'enable_market_structure_patterns' => isset($_POST['enable_market_structure_patterns']) ? 1 : 0,
        'enable_volume_profile_patterns' => isset($_POST['enable_volume_profile_patterns']) ? 1 : 0,
        'alert_filtering_enabled' => isset($_POST['alert_filtering_enabled']) ? 1 : 0,
        'max_alerts_per_coin' => intval($_POST['max_alerts_per_coin'] ?? 50),
        'group_alerts_by_coin' => isset($_POST['group_alerts_by_coin']) ? 1 : 0
    ];
    
    // Handle alert types
    $alertTypes = [];
    $defaultTypes = [
        'rsi', 'price', 'support', 'orderbook', 'liquidation',
        'multi', 'volume', 'funding', 'oi_change', 'whale',
        'breakout', 'second_rejection', 'two_percent_drop',
        'buying_spike', 'selling_spike', 'accumulation',
        'distribution', 'volume_climax', 'pattern'
    ];
    
    foreach ($defaultTypes as $type) {
        $alertTypes[$type] = isset($_POST["alert_type_{$type}"]) ? 1 : 0;
    }
    
    $settings['alertTypes'] = $alertTypes;
    
    $db->updateUserSettings($userId, $settings);
    
    $_SESSION['flash'] = ['type' => 'success', 'message' => 'Settings saved successfully!'];
    header('Location: settings.php');
    exit;
}

// Get current settings
$settings = $db->getUserSettings($userId);
$csrfToken = generateCSRFToken();
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Settings - Binance Futures Alert System</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <nav class="navbar">
        <div class="nav-brand">Binance Futures Alerts</div>
        <div class="nav-menu">
            <a href="dashboard.php">Dashboard</a>
            <a href="alerts.php">Alerts</a>
            <a href="tracked-coins.php">Tracked Coins</a>
            <a href="settings.php" class="active">Settings</a>
            <a href="logout.php">Logout (<?php echo htmlspecialchars($_SESSION['username']); ?>)</a>
        </div>
    </nav>
    
    <div class="container">
        <h1>Settings</h1>
        
        <?php if (isset($_SESSION['flash'])): ?>
            <div class="alert alert-<?php echo $_SESSION['flash']['type']; ?>">
                <?php echo $_SESSION['flash']['message']; ?>
            </div>
            <?php unset($_SESSION['flash']); ?>
        <?php endif; ?>
        
        <form method="POST" class="settings-form">
            <input type="hidden" name="csrf_token" value="<?php echo $csrfToken; ?>">
            
            <div class="settings-section">
                <h2>General Settings</h2>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="enabled" <?php echo $settings['enabled'] ? 'checked' : ''; ?>>
                        Enable Monitoring
                    </label>
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="show_notifications" <?php echo $settings['show_notifications'] ? 'checked' : ''; ?>>
                        Show Browser Notifications
                    </label>
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="send_emails" <?php echo $settings['send_emails'] ? 'checked' : ''; ?>>
                        Send Email Alerts
                    </label>
                </div>
                
                <div class="form-group">
                    <label>Notification Cooldown (seconds)</label>
                    <input type="number" name="notification_cooldown" value="<?php echo $settings['notification_cooldown']; ?>" min="10" max="3600">
                </div>
                
                <div class="form-group">
                    <label>Minimum 24h Volume (USD)</label>
                    <input type="number" name="min_volume" value="<?php echo $settings['min_volume']; ?>" min="100000" max="1000000000" step="100000">
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Alert Types</h2>
                
                <div class="alert-types-grid">
                    <?php
                    $alertTypeLabels = [
                        'rsi' => 'RSI Alerts',
                        'price' => 'Price Alerts',
                        'support' => 'Support/Resistance',
                        'orderbook' => 'Order Book',
                        'liquidation' => 'Liquidations',
                        'multi' => 'Multi-Condition',
                        'volume' => 'Volume',
                        'funding' => 'Funding Rate',
                        'oi_change' => 'Open Interest',
                        'whale' => 'Whale Orders',
                        'breakout' => 'Breakouts',
                        'second_rejection' => 'Second Rejection',
                        'two_percent_drop' => '2% Drop',
                        'buying_spike' => 'Buying Spike',
                        'selling_spike' => 'Selling Spike',
                        'accumulation' => 'Accumulation',
                        'distribution' => 'Distribution',
                        'volume_climax' => 'Volume Climax',
                        'pattern' => 'Patterns'
                    ];
                    
                    foreach ($alertTypeLabels as $type => $label):
                    ?>
                    <div class="form-group checkbox">
                        <label>
                            <input type="checkbox" name="alert_type_<?php echo $type; ?>" 
                                   <?php echo ($settings['alertTypes'][$type] ?? true) ? 'checked' : ''; ?>>
                            <?php echo $label; ?>
                        </label>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <div class="settings-section">
                <h2>RSI Settings</h2>
                
                <div class="form-group">
                    <label>RSI Check Interval (seconds)</label>
                    <input type="number" name="rsi_check_interval" value="<?php echo $settings['rsi_check_interval']; ?>" min="10" max="300">
                </div>
                
                <div class="form-group">
                    <label>RSI Alert Threshold</label>
                    <input type="number" name="rsi_threshold" value="<?php echo $settings['rsi_threshold']; ?>" min="50" max="100">
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="rsi7_15m_enabled" <?php echo $settings['rsi7_15m_enabled'] ? 'checked' : ''; ?>>
                        Enable RSI 7 (15m) alerts
                    </label>
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="rsi7_1d_enabled" <?php echo $settings['rsi7_1d_enabled'] ? 'checked' : ''; ?>>
                        Enable RSI 7 (1D) alerts
                    </label>
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Price Alert Settings</h2>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="price_alerts_enabled" <?php echo $settings['price_alerts_enabled'] ? 'checked' : ''; ?>>
                        Enable Price Alerts
                    </label>
                </div>
                
                <div class="form-group">
                    <label>24H High Break Threshold (%)</label>
                    <input type="number" name="price_break_threshold" value="<?php echo $settings['price_break_threshold']; ?>" min="0.1" max="5" step="0.1">
                </div>
                
                <div class="form-group">
                    <label>Price Increase Threshold (%)</label>
                    <input type="number" name="price_increase_threshold" value="<?php echo $settings['price_increase_threshold']; ?>" min="1" max="50">
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Order Book Settings</h2>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="order_book_alerts_enabled" <?php echo $settings['order_book_alerts_enabled'] ? 'checked' : ''; ?>>
                        Enable Order Book Alerts
                    </label>
                </div>
                
                <div class="form-group">
                    <label>Order Book Interval (minutes)</label>
                    <input type="number" name="order_book_interval" value="<?php echo $settings['order_book_interval']; ?>" min="1" max="60">
                </div>
                
                <div class="form-group">
                    <label>Order Book Depth</label>
                    <input type="number" name="order_book_depth" value="<?php echo $settings['order_book_depth']; ?>" min="10" max="100">
                </div>
                
                <div class="form-group">
                    <label>Bearish Threshold (%)</label>
                    <input type="number" name="order_book_bearish_threshold" value="<?php echo $settings['order_book_bearish_threshold']; ?>" min="-50" max="0">
                </div>
                
                <div class="form-group">
                    <label>Bullish Threshold (%)</label>
                    <input type="number" name="order_book_bullish_threshold" value="<?php echo $settings['order_book_bullish_threshold']; ?>" min="0" max="50">
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Volume Spike Detection</h2>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="directional_volume_spike_enabled" <?php echo $settings['directional_volume_spike_enabled'] ? 'checked' : ''; ?>>
                        Enable Volume Spike Detection
                    </label>
                </div>
                
                <div class="form-group">
                    <label>Buy Volume Spike Threshold (x)</label>
                    <input type="number" name="buy_volume_spike_threshold" value="<?php echo $settings['buy_volume_spike_threshold']; ?>" min="2" max="10" step="0.5">
                </div>
                
                <div class="form-group">
                    <label>Sell Volume Spike Threshold (x)</label>
                    <input type="number" name="sell_volume_spike_threshold" value="<?php echo $settings['sell_volume_spike_threshold']; ?>" min="2" max="10" step="0.5">
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="accumulation_distribution_enabled" <?php echo $settings['accumulation_distribution_enabled'] ? 'checked' : ''; ?>>
                        Enable Accumulation/Distribution Analysis
                    </label>
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Liquidation Settings</h2>
                
                <div class="form-group">
                    <label>Liquidation Check Interval (minutes)</label>
                    <input type="number" name="liquidation_interval" value="<?php echo $settings['liquidation_interval']; ?>" min="1" max="10">
                </div>
                
                <div class="form-group">
                    <label>Liquidation Alert Threshold (USD)</label>
                    <input type="number" name="liquidation_alert_threshold" value="<?php echo $settings['liquidation_alert_threshold']; ?>" min="100000" max="10000000" step="100000">
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Whale Alert Settings</h2>
                
                <div class="form-group">
                    <label>Whale Order Threshold (USD)</label>
                    <input type="number" name="whale_order_threshold" value="<?php echo $settings['whale_order_threshold']; ?>" min="100000" max="10000000" step="100000">
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Pattern Recognition</h2>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="pattern_recognition_enabled" <?php echo $settings['pattern_recognition_enabled'] ? 'checked' : ''; ?>>
                        Enable Pattern Recognition
                    </label>
                </div>
                
                <div class="form-group">
                    <label>Minimum Pattern Confidence (%)</label>
                    <input type="number" name="min_pattern_confidence" value="<?php echo $settings['min_pattern_confidence']; ?>" min="30" max="100">
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="enable_candlestick_patterns" <?php echo $settings['enable_candlestick_patterns'] ? 'checked' : ''; ?>>
                        Enable Candlestick Patterns
                    </label>
                </div>
            </div>
            
            <div class="settings-section">
                <h2>Alert Filtering</h2>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="alert_filtering_enabled" <?php echo $settings['alert_filtering_enabled'] ? 'checked' : ''; ?>>
                        Enable Alert Filtering
                    </label>
                </div>
                
                <div class="form-group">
                    <label>Max Alerts Per Coin</label>
                    <input type="number" name="max_alerts_per_coin" value="<?php echo $settings['max_alerts_per_coin']; ?>" min="10" max="200">
                </div>
                
                <div class="form-group checkbox">
                    <label>
                        <input type="checkbox" name="group_alerts_by_coin" <?php echo $settings['group_alerts_by_coin'] ? 'checked' : ''; ?>>
                        Group Alerts by Coin
                    </label>
                </div>
            </div>
            
            <div class="form-actions">
                <button type="submit" class="btn btn-primary">Save Settings</button>
            </div>
        </form>
    </div>
</body>
</html>