<?php
// public/api/get_data.php

session_start();
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/Database.php';
require_once __DIR__ . '/../../includes/AlertChecker.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Not authenticated']);
    exit;
}

$userId = $_SESSION['user_id'];
$db = Database::getInstance();
$action = $_GET['action'] ?? '';

switch ($action) {
    case 'stats':
        $pdo = $db->getConnection();
        
        // Get RSI count
        $stmt = $pdo->query("SELECT COUNT(DISTINCT symbol) as count FROM rsi_data WHERE recorded_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)");
        $rsiCount = $stmt->fetch()['count'];
        
        // Get last update time
        $stmt = $pdo->query("SELECT MAX(recorded_at) as last FROM rsi_data");
        $lastUpdate = $stmt->fetch()['last'];
        
        echo json_encode([
            'success' => true,
            'rsiCount' => $rsiCount,
            'lastUpdate' => $lastUpdate ? formatTimeAgo($lastUpdate) : 'Never'
        ]);
        break;
        
    case 'analyze':
        try {
            $checker = new AlertChecker($userId);
            $checker->checkAllAlerts();
            
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
        break;
        
    case 'rsi_data':
        $pdo = $db->getConnection();
        $stmt = $pdo->prepare("
            SELECT * FROM rsi_data 
            WHERE recorded_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
            ORDER BY volume_24h DESC
            LIMIT 100
        ");
        $stmt->execute();
        $data = $stmt->fetchAll();
        
        echo json_encode(['success' => true, 'data' => $data]);
        break;
        
    case 'tracked_coins':
        $trackedCoins = $db->getTrackedCoins($userId);
        echo json_encode(['success' => true, 'data' => $trackedCoins]);
        break;
        
    case 'add_tracked_coin':
        $symbol = $_POST['symbol'] ?? '';
        if (!$symbol) {
            echo json_encode(['success' => false, 'error' => 'Symbol required']);
            exit;
        }
        
        // Get current price
        require_once __DIR__ . '/../../includes/BinanceAPI.php';
        $api = new BinanceAPI();
        
        try {
            $price = $api->getPrice($symbol);
            $ticker = $api->get24hrTicker();
            
            $tickerData = null;
            foreach ($ticker as $t) {
                if ($t['symbol'] === $symbol) {
                    $tickerData = $t;
                    break;
                }
            }
            
            if ($price && $tickerData) {
                $data = [
                    'current_price' => floatval($price),
                    'volume_24h' => floatval($tickerData['quoteVolume']),
                    'trend' => 'neutral',
                    'price_action' => 'consolidation',
                    'volatility' => 0,
                    'volume_ratio' => 1,
                    'supportLevels' => [
                        ['level' => floatval($price) * 0.98, 'strength' => 5],
                        ['level' => floatval($price) * 0.95, 'strength' => 3]
                    ],
                    'resistanceLevels' => [
                        ['level' => floatval($price) * 1.02, 'strength' => 5],
                        ['level' => floatval($price) * 1.05, 'strength' => 3]
                    ]
                ];
                
                $db->addTrackedCoin($userId, $symbol, $data);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Could not fetch price']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
        break;
        
    case 'remove_tracked_coin':
        $symbol = $_POST['symbol'] ?? '';
        if (!$symbol) {
            echo json_encode(['success' => false, 'error' => 'Symbol required']);
            exit;
        }
        
        $db->removeTrackedCoin($userId, $symbol);
        echo json_encode(['success' => true]);
        break;
        
    case 'notifications':
        require_once __DIR__ . '/../../includes/NotificationService.php';
        $notifier = new NotificationService();
        $notifications = $notifier->getBrowserNotifications($userId);
        
        echo json_encode(['success' => true, 'notifications' => $notifications]);
        break;
        
    default:
        echo json_encode(['error' => 'Invalid action']);
}