<?php
// includes/BinanceAPI.php

class BinanceAPI {
    private $baseUrl;
    private $timeout = 30;
    
    public function __construct() {
        $this->baseUrl = $_ENV['BINANCE_API_URL'] ?? 'https://fapi.binance.com';
    }
    
    /**
     * Get futures exchange info
     */
    public function getExchangeInfo() {
        return $this->request('/fapi/v1/exchangeInfo');
    }
    
    /**
     * Get 24hr ticker for all symbols
     */
    public function get24hrTicker() {
        return $this->request('/fapi/v1/ticker/24hr');
    }
    
    /**
     * Get klines/candles
     */
    public function getKlines($symbol, $interval = '15m', $limit = 100) {
        return $this->request('/fapi/v1/klines', [
            'symbol' => $symbol,
            'interval' => $interval,
            'limit' => $limit
        ]);
    }
    
    /**
     * Get order book
     */
    public function getOrderBook($symbol, $limit = 50) {
        return $this->request('/fapi/v1/depth', [
            'symbol' => $symbol,
            'limit' => $limit
        ]);
    }
    
    /**
     * Get current price
     */
    public function getPrice($symbol) {
        $data = $this->request('/fapi/v1/ticker/price', ['symbol' => $symbol]);
        return $data['price'] ?? null;
    }
    
    /**
     * Get funding rate
     */
    public function getFundingRate($symbol) {
        $data = $this->request('/fapi/v1/fundingRate', [
            'symbol' => $symbol,
            'limit' => 1
        ]);
        return isset($data[0]) ? floatval($data[0]['fundingRate']) * 100 : null;
    }
    
    /**
     * Get open interest
     */
    public function getOpenInterest($symbol) {
        return $this->request('/fapi/v1/openInterest', ['symbol' => $symbol]);
    }
    
    /**
     * Get liquidation orders
     */
    public function getForceOrders($startTime = null, $limit = 200) {
        $params = ['limit' => $limit];
        if ($startTime) {
            $params['startTime'] = $startTime;
        }
        return $this->request('/fapi/v1/allForceOrders', $params);
    }
    
    /**
     * Make API request
     */
    private function request($endpoint, $params = []) {
        $url = $this->baseUrl . $endpoint;
        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_error($ch)) {
            throw new Exception('CURL Error: ' . curl_error($ch));
        }
        
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("HTTP Error: $httpCode - $response");
        }
        
        $data = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('JSON Parse Error: ' . json_last_error_msg());
        }
        
        return $data;
    }
}